/*
 * Copyright © 2025 Jelurida Swiss SA
 *
 * See the LICENSE.txt file at the top-level directory of this distribution
 * for licensing information.
 *
 * Unless otherwise agreed in a custom licensing agreement with Jelurida
 * Swiss SA, no part of this software, including this file, may be copied,
 * modified, propagated, or distributed except according to the terms
 * contained in the LICENSE.txt file.
 *
 * Removal or modification of this copyright notice is prohibited.
 *
 */

package nxt.addons;

import nxt.Account;
import nxt.Block;
import nxt.BlockchainProcessor;
import nxt.Constants;
import nxt.Db;
import nxt.Nxt;
import nxt.util.Listener;

import java.sql.Connection;
import java.sql.PreparedStatement;
import java.sql.ResultSet;
import java.sql.SQLException;
import java.util.Collections;
import java.util.List;
import java.util.Map;
import java.util.SortedMap;
import java.util.TreeMap;

/**
 * An addon that creates a snapshot for the migration of Nxt to an Ardor child chain.
 */
public class ChildChainSnapshot implements AddOn {
    private static final int snapshotHeight = Nxt.getIntProperty("nxt.childChainSnapshotHeight", Integer.MAX_VALUE);
    private static final boolean snapshotForTestnet = Nxt.getBooleanProperty("nxt.childChainSnapshotForTestnet", true);

    @Override
    public void init() {
        Nxt.getBlockchainProcessor().addListener(new Listener<Block>() {

            private final List<byte[]> developerPublicKeys = snapshotForTestnet ? SnapshotUtils.loadDeveloperPublicKeys() : Collections.emptyList();

            @Override
            public void notify(Block block) {
                if (block.getHeight() == snapshotHeight) {
                    exportPublicKeys();
                    exportBalances();
                    exportAliases();
                    exportCurrencies();
                }
            }

            private void exportPublicKeys() {
                String file = snapshotForTestnet ? "NXT_PUBLIC_KEY-testnet.json" : "NXT_PUBLIC_KEY.json";
                SnapshotUtils.exportPublicKeys(developerPublicKeys, file);
            }

            private void exportBalances() {
                SortedMap<String, Long> snapshotMap = new TreeMap<>();
                try ( Connection con = Db.db.getConnection();
                      PreparedStatement pstmt = con.prepareStatement("SELECT id, balance FROM account WHERE LATEST=true")) {
                    try ( ResultSet rs = pstmt.executeQuery() ) {
                        while (rs.next()) {
                            long accountId = rs.getLong("id");
                            long balance = rs.getLong("balance");
                            if (balance <= 0) {
                                continue;
                            }
                            // if for testnet, reduce by 2 to reserve for developer accounts
                            if (snapshotForTestnet && !developerPublicKeys.isEmpty()) {
                                balance = balance / 2;
                            }
                            String account = Long.toUnsignedString(accountId);
                            snapshotMap.put(account, balance);
                        }
                    }
                } catch (SQLException e) {
                    throw new RuntimeException(e.getMessage(), e);
                }
                if (snapshotForTestnet && !developerPublicKeys.isEmpty()) {
                    final long developerBalance = Constants.MAX_BALANCE_NQT / (2L * developerPublicKeys.size());
                    developerPublicKeys.forEach(publicKey -> {
                        String account = Long.toUnsignedString(Account.getId(publicKey));
                        snapshotMap.put(account, developerBalance);
                    });
                }
                SnapshotUtils.saveMap(snapshotMap, snapshotForTestnet ? "NXT-testnet.json" : "NXT.json");
            }

            private void exportAliases() {
                SortedMap<String, Map<String, String>> snapshotMap = SnapshotUtils.createAliasSnapshot();
                SnapshotUtils.saveMap(snapshotMap, snapshotForTestnet ? "NXT_ALIASES-testnet.json" : "NXT_ALIASES.json");
            }

            private void exportCurrencies() {
                SortedMap<String, Map<String, String>> snapshotMap = SnapshotUtils.createCurrenciesSnapshot();
                SnapshotUtils.saveMap(snapshotMap, snapshotForTestnet ? "NXT_CURRENCIES-testnet.json" : "NXT_CURRENCIES.json");
            }

        }, BlockchainProcessor.Event.AFTER_BLOCK_ACCEPT);
    }
}
