/*
 * Copyright © 2013-2016 The Nxt Core Developers.
 * Copyright © 2016-2023 Jelurida IP B.V.
 * Copyright © 2023-2025 Jelurida Swiss SA
 *
 * See the LICENSE.txt file at the top-level directory of this distribution
 * for licensing information.
 *
 * Unless otherwise agreed in a custom licensing agreement with Jelurida
 * Swiss SA, no part of this software, including this file, may be copied,
 * modified, propagated, or distributed except according to the terms
 * contained in the LICENSE.txt file.
 *
 * Removal or modification of this copyright notice is prohibited.
 *
 */

package nxt.addons;

import nxt.Account;
import nxt.Db;
import nxt.crypto.Crypto;
import nxt.util.Convert;
import nxt.util.JSON;
import nxt.util.Logger;
import org.json.simple.JSONArray;

import java.io.BufferedReader;
import java.io.BufferedWriter;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.InputStreamReader;
import java.io.OutputStreamWriter;
import java.io.PrintWriter;
import java.sql.Connection;
import java.sql.PreparedStatement;
import java.sql.ResultSet;
import java.sql.SQLException;
import java.util.ArrayList;
import java.util.Collections;
import java.util.List;
import java.util.Locale;
import java.util.Map;
import java.util.SortedMap;
import java.util.TreeMap;

class SnapshotUtils {

    static List<byte[]> loadDeveloperPublicKeys() {
        List<byte[]> developerPublicKeys = new ArrayList<>();
        InputStream is = ClassLoader.getSystemResourceAsStream("developerPasswords.txt");
        if (is != null) {
            try ( BufferedReader reader = new BufferedReader(new InputStreamReader(is))) {
                String line;
                while ((line = reader.readLine()) != null) {
                    developerPublicKeys.add(Crypto.getPublicKey(line));
                }
            } catch (IOException e) {
                throw new RuntimeException(e.getMessage(), e);
            }
            developerPublicKeys.sort(Convert.byteArrayComparator);
            Logger.logDebugMessage("Loaded " + developerPublicKeys.size() + " developer public keys");
        } else {
            Logger.logDebugMessage("No developerPasswords.txt file found");
        }
        return developerPublicKeys;
    }

    static void exportPublicKeys(List<byte[]> developerPublicKeys, String file) {
        JSONArray publicKeys = new JSONArray();
        try ( Connection con = Db.db.getConnection();
              PreparedStatement pstmt = con.prepareStatement("SELECT public_key FROM public_key WHERE public_key IS NOT NULL AND LATEST=true ORDER by account_id")) {
            try (ResultSet rs = pstmt.executeQuery()) {
                while (rs.next()) {
                    byte[] publicKey = rs.getBytes("public_key");
                    if (Collections.binarySearch(developerPublicKeys, publicKey, Convert.byteArrayComparator) >= 0) {
                        throw new RuntimeException("Developer account " + Account.getId(publicKey) + " already exists");
                    }
                    publicKeys.add(Convert.toHexString(rs.getBytes("public_key")));
                }
            }
        } catch (SQLException e) {
            throw new RuntimeException(e.getMessage(), e);
        }
        developerPublicKeys.forEach(publicKey -> publicKeys.add(Convert.toHexString(publicKey)));
        Logger.logInfoMessage("Will save " + publicKeys.size() + " public keys");
        try ( PrintWriter writer = new PrintWriter((new BufferedWriter(new OutputStreamWriter(new FileOutputStream(file)))), true)) {
            JSON.writeJSONString(publicKeys, writer);
        } catch (IOException e) {
            throw new RuntimeException(e.getMessage(), e);
        }
        Logger.logInfoMessage("Done");
    }

    static void saveMap(Map<String, ?> snapshotMap, String file) {
        Logger.logInfoMessage("Will save " + snapshotMap.size() + " entries to " + file);
        try ( PrintWriter writer = new PrintWriter((new BufferedWriter(new OutputStreamWriter(new FileOutputStream(file)))), true)) {
            StringBuilder sb = new StringBuilder(1024);
            JSON.encodeObject(snapshotMap, sb);
            writer.write(sb.toString());
        } catch (IOException e) {
            throw new RuntimeException(e.getMessage(), e);
        }
        Logger.logInfoMessage("Done");
    }

    static SortedMap<String, Map<String, String>> createAliasSnapshot() {
        SortedMap<String, Map<String, String>> snapshotMap = new TreeMap<>();
        try ( Connection con = Db.db.getConnection();
              PreparedStatement pstmt = con.prepareStatement("SELECT account_id, alias_name, alias_uri FROM alias WHERE LATEST=true")) {
            try ( ResultSet rs = pstmt.executeQuery()) {
                while (rs.next()) {
                    String aliasName = rs.getString("alias_name");
                    String aliasURI = Convert.nullToEmpty(rs.getString("alias_uri"));
                    long accountId = rs.getLong("account_id");
                    Map<String, String> alias = new TreeMap<>();
                    alias.put("account", Long.toUnsignedString(accountId));
                    alias.put("uri", aliasURI);
                    snapshotMap.put(aliasName, alias);
                }
            }
        } catch (SQLException e) {
            throw new RuntimeException(e.getMessage(), e);
        }
        return snapshotMap;
    }

    static SortedMap<String, Map<String, String>> createCurrenciesSnapshot() {
        SortedMap<String, Map<String, String>> snapshotMap = new TreeMap<>();
        try (Connection con = Db.db.getConnection();
             PreparedStatement pstmt = con.prepareStatement("SELECT account_id, name, code FROM currency WHERE LATEST=true")) {
            try (ResultSet rs = pstmt.executeQuery()) {
                while (rs.next()) {
                    String currencyName = rs.getString("name");
                    String currencyCode = rs.getString("code");
                    if (invalidCurrency(currencyCode, currencyName.toLowerCase(Locale.ROOT))) {
                        Logger.logDebugMessage("Skipping currency " + currencyCode + " " + currencyName);
                        continue;
                    }
                    long accountId = rs.getLong("account_id");
                    Map<String, String> currency = new TreeMap<>();
                    currency.put("account", Long.toUnsignedString(accountId));
                    currency.put("name", currencyName);
                    snapshotMap.put(currencyCode, currency);
                }
            }
        } catch (SQLException e) {
            throw new RuntimeException(e.getMessage(), e);
        }
        return snapshotMap;
    }

    private static boolean invalidCurrency(String code, String normalizedName) {
        if (code.equals("ARDOR") || code.contains("ARDR") || "ardor".equals(normalizedName) || "ardr".equals(normalizedName)) {
            return true;
        }
        if (code.contains("NXT") || code.contains("NEXT") || "nxt".equals(normalizedName) || "next".equals(normalizedName)) {
            return true;
        }
        if (code.equals("IGNIS") || "ignis".equals(normalizedName)) {
            return true;
        }
        if ("bitswift".equals(normalizedName)) {
            return true;
        }
        if (code.equals("AEUR") || "aeur".equals(normalizedName)) {
            return true;
        }
        return false;
    }
}
